<?php
/**
 * LocalizedDate.php 2020-4-9
 * Gambio GmbH
 * http://www.gambio.de
 * Copyright (c) 2020 Gambio GmbH
 * Released under the GNU General Public License (Version 2)
 * [http://www.gnu.org/licenses/gpl-2.0.html]
 */

declare(strict_types=1);

/**
 * Class LocalizedDate
 */
class LocalizedDate
{
    /**
     * @var string
     */
    protected $format;
    
    /**
     * @var string
     */
    protected $languageCode;
    
    /**
     * @var int|null
     */
    protected $timestamp;
    
    
    /**
     * LocalizedDate constructor.
     *
     * @param string   $format
     * @param string   $languageCode
     * @param int|null $timestamp
     */
    public function __construct(string $format, string $languageCode, ?int $timestamp = null)
    {
        $this->format       = $format;
        $this->languageCode = $languageCode;
        $this->timestamp    = $timestamp;
    }
    
    
    /**
     * @return string
     */
    public function __toString(): string
    {
        $this->setLocale($this->languageCode);
        
        $result = strftime($this->format, $this->timestamp ?? time());
        $result = utf8_encode_wrapper($result);
        
        $this->setLocale(null);
        
        return $result;
    }
    
    
    /**
     * @param string|null $languageCode
     */
    protected function setLocale(?string $languageCode): void
    {
        if ($languageCode !== null) {
            
            $locale = $this->getLocaleByLanguageCode($languageCode);
        }
        
        if ($languageCode === null || $this->localeIsInstalled($locale) === false) {
            
            $locale = Locale::getDefault();
        }
        
        setlocale(LC_TIME, $locale);
    }
    
    
    /**
     * @param string $code
     *
     * @return string
     */
    protected function getLocaleByLanguageCode(string $code): string
    {
        switch (strtolower($code)) {
            
            case 'de' :
                $locale = 'de_DE';
                break;
            
            case 'en' :
                $locale = 'en_US';
                break;
            
            default :
                $locale = Locale::acceptFromHttp($_SERVER['HTTP_ACCEPT_LANGUAGE']);
                break;
        }
        
        return $locale;
    }
    
    
    /**
     * @param string $locale
     *
     * @return bool
     */
    protected function localeIsInstalled(string $locale): bool
    {
        $installedLocales = ResourceBundle::getLocales('');
        
        return in_array($locale, $installedLocales, true);
    }
}